/*
 * MeleeAttackEffectInfo.h
 *
 * Created 9/6/2009 By Johnny Huynh
 *
 * Version 00.00.02 9/6/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef MELEE_ATTACK_EFFECT_INFO_H
 #define MELEE_ATTACK_EFFECT_INFO_H
 
 template <typename T> class MeleeAttackEffectInfo;
 
 #include "global.h"
 
 #include "collisionEntry.h"
 #include "EffectInfo.h"
 #include "EffectTask.h"
 #include "pointerTo.h"
 
 #include "MeleeAttackInfo.h"
 #include "PushBackEffectInfo.h"
 
 /**
  * Class specification for AttackEffectInfo
  */
 template <typename T>
 class MeleeAttackEffectInfo : public PushEffectInfo<T>
 {
 // Protected Static Functions
 protected:
    static inline void process_melee_attack_effect( EffectInfo<T>* effect_info_Ptr, Object<T>* into_object_Ptr, const CollisionEntry& c_entry );
 
 // Data Members
 private:
    PT(DamageObject<T>) _dmg_obj_Ptr;
    double _duration_before_attaching_dmg_obj;
    double _duration_before_detaching_dmg_obj; //_duration_before_detaching_dmg_obj_after_attachment;
    
 // Local Functions
 public:
    MeleeAttackEffectInfo( DamageObject<T>* dmg_obj_Ptr = NULL, 
                           const double duration_before_attaching_dmg_obj = 0.0, 
                           const double duration_before_detaching_dmg_obj = 0.0,
                           const double stun_duration = 0.0, const T move_distance = ZERO, 
                           const double move_duration = 0.0, 
                           EffectFunc* effect_func = &MeleeAttackEffectInfo<T>::process_melee_attack_effect );
    MeleeAttackEffectInfo( const MeleeAttackEffectInfo<T>& melee_attack_effect_info );
    virtual ~MeleeAttackEffectInfo();
    inline MeleeAttackEffectInfo<T>& operator=( const MeleeAttackEffectInfo<T>& melee_attack_effect_info );
    inline DamageObject<T>* get_damage_object();
    inline double get_duration_before_attaching_damage_object() const;
    inline double get_duration_before_detaching_damage_object() const;
 
 // Private Functions
 private:
    
 // Public Static Functions
 public:
    
 };
 
 /** STATIC FUNCTIONS **/
 
 /**
  * process_melee_attack_effect() applies the specified melee attack effect pointed to onto the specified object pointed to.
  * The effect_info_Ptr must be a pointer to a MeleeAttackEffectInfo.
  *
  * @param (EffectInfo<T>*) effect_info_Ptr
  * @param (Object<T>*) into_object_Ptr
  * @param (const CollisionEntry&) c_entry
  */
 template <typename T>
 inline void MeleeAttackEffectInfo<T>::process_melee_attack_effect( EffectInfo<T>* effect_info_Ptr, 
                                                            Object<T>* into_object_Ptr, const CollisionEntry& c_entry )
 {  
    nassertv( effect_info_Ptr != NULL );
    nassertv( into_object_Ptr != NULL );
    
    PushEffectInfo<T>::process_push_back_effect( effect_info_Ptr, into_object_Ptr, c_entry );
    
    MeleeAttackEffectInfo<T>* melee_attack_effect_info_Ptr( dynamic_cast<MeleeAttackEffectInfo<T>*>(effect_info_Ptr) );
    
    PT(MeleeAttackInfo<T>) melee_attack_info( new MeleeAttackInfo<T>(
                                               melee_attack_effect_info_Ptr->get_damage_object(),
                                               melee_attack_effect_info_Ptr->get_duration_before_attaching_damage_object(),
                                               melee_attack_effect_info_Ptr->get_duration_before_detaching_damage_object()) );
    PT(EffectTask<T>) effect_task( new EffectTask<T>( into_object_Ptr, melee_attack_info ) );
    effect_task->add_to_task_manager();
 }
 
 /** LOCAL FUNCTIONS **/
 
 /**
  * Constructor
  */
 template <typename T>
 MeleeAttackEffectInfo<T>::MeleeAttackEffectInfo( DamageObject<T>* dmg_obj_Ptr, 
                                                  const double duration_before_attaching_dmg_obj, 
                                                  const double duration_before_detaching_dmg_obj,
                                                  const double stun_duration, 
                                                  const T move_distance, const double move_duration,
                                                  EffectFunc* effect_func )
                     : PushEffectInfo<T>( stun_duration, move_distance, move_duration, effect_func ),
                       _dmg_obj_Ptr( dmg_obj_Ptr ),
                       _duration_before_attaching_dmg_obj( duration_before_attaching_dmg_obj ),
                       _duration_before_detaching_dmg_obj( duration_before_detaching_dmg_obj )
 {
    
 }
 
 /**
  * Copy Constructor
  */
 template <typename T>
 MeleeAttackEffectInfo<T>::MeleeAttackEffectInfo( const MeleeAttackEffectInfo<T>& melee_attack_effect_info )
                          : PushEffectInfo<T>( melee_attack_effect_info ),
                            _dmg_obj_Ptr( melee_attack_effect_info._dmg_obj_Ptr ),
                            _duration_before_attaching_dmg_obj( melee_attack_effect_info._duration_before_attaching_dmg_obj ),
                            _duration_before_detaching_dmg_obj( melee_attack_effect_info._duration_before_detaching_dmg_obj )
 {
    
 }
 
 /**
  * Destructor
  */
 template <typename T>
 MeleeAttackEffectInfo<T>::~MeleeAttackEffectInfo()
 {
    
 }
 
 /**
  * operator=() copies the content of the specified MeleeAttackEffectInfo to this MeleeAttackEffectInfo.
  *
  * @param (const MeleeAttackEffectInfo<T>&) melee_attack_effect_info
  * @return MeleeAttackEffectInfo<T>&
  */
 template <typename T>
 inline MeleeAttackEffectInfo<T>& MeleeAttackEffectInfo<T>::operator=( 
                                                                const MeleeAttackEffectInfo<T>& melee_attack_effect_info )
 {
    PushEffectInfo<T>::operator=( melee_attack_effect_info );
    _dmg_obj_Ptr = melee_attack_effect_info._dmg_obj_Ptr;
    _duration_before_attaching_dmg_obj = melee_attack_effect_info._duration_before_attaching_dmg_obj;
    _duration_before_detaching_dmg_obj = melee_attack_effect_info._duration_before_detaching_dmg_obj;
    
    return *this;
 }
 
 /**
  * get_damage_object() returns a pointer to the DamageObject.
  *
  * @return double
  */
 template <typename T>
 inline DamageObject<T>* MeleeAttackEffectInfo<T>::get_damage_object()
 {
    return _dmg_object_Ptr
 }
 
 /**
  * get_duration_before_attaching_damage_object() returns the duration before attaching 
  * the damage object.
  *
  * @return double
  */
 template <typename T>
 inline double MeleeAttackEffectInfo<T>::get_duration_before_attaching_damage_object() const
 {
    return _duration_before_attaching_dmg_obj;
 }
 
 /**
  * get_duration_before_detaching_damage_object() returns the duration before detaching 
  * the damage object.
  *
  * @return double
  */
 template <typename T>
 inline double MeleeAttackEffectInfo<T>::get_duration_before_detaching_damage_object() const
 {
    return _duration_before_detaching_dmg_obj;
 }
 
 /** PUBLIC STATIC FUNCTIONS **/
 
 #endif // MELEE_ATTACK_EFFECT_INFO_H